<?php
/**
 * Ninja functions and definitions
 *
 * @package ninja
 */

 /**
 * Set the content width based on the theme's design and stylesheet.
 * Used to set the width of images and content. Should be equal to the width the theme is designed for, generally via the style.css stylesheet.
 */
if ( ! isset( $content_width ) ) { $content_width = 648; /* pixels */ }

/**
 * Set the content width for full width pages with no sidebar.
 */
function ninja_content_width() {
  if ( is_page_template( 'page-fullwidth.php' ) ) {
    global $content_width;
    $content_width = 1008; /* pixels */
  }
}
add_action( 'template_redirect', 'ninja_content_width' );

/**
 * Set container based on template
 */
function set_ninja_container(){
    $container = [];
    $templates = [];

    $templates = ['page-fullwidth.php', 'page-home.php'];

    if ( is_page_template($templates) ) {
        $container['container_type'] = 'container-fluid';
    }else{
        $container['container_type'] = 'container';
    }

    return $container;
}

function ninja_container(){
    $container = set_ninja_container();
    $ninja_container = $container['container_type'];
    return $ninja_container;
}

if ( ! function_exists( 'ninja_setup' ) ) :
/**
 * Sets up theme defaults and registers support for various WordPress features.
 *
 * Note that this function is hooked into the after_setup_theme hook, which
 * runs before the init hook. The init hook is too late for some features, such
 * as indicating support for post thumbnails.
 */
function ninja_setup() {
    // Make theme available for translation. Translations can be filed in the /languages/ directory.
    //load_theme_textdomain( 'ninja', get_template_directory() . '/languages' );
    // Enable support for Post Thumbnails on posts and pages.
    add_theme_support( 'post-thumbnails' );
    // This theme uses wp_nav_menu() in one location.
    register_nav_menus( array(
        'primary'      => esc_html__( 'Primary Menu', 'ninja' ),
        'social-menu'  => esc_html__( 'Social Menu', 'ninja' )
    ) );
    // Enable support for HTML5 markup.
    add_theme_support( 'html5', array(
        'search-form',
        'gallery',
        'caption'
    ) );
    // Let WordPress manage the document title in the document head.
    add_theme_support( 'title-tag' );
    // Enable support for WooCommerce.
    add_theme_support( 'woocommerce' );
    // Enable support for custom header
    register_default_headers( array(
        'default-image' => array(
            'url'           => get_stylesheet_directory_uri() . '/images/logo2.jpg',
            'thumbnail_url' => get_stylesheet_directory_uri() . '/images/logo2.jpg',
            'description'   => __( 'Default Logo', 'ninja' )
        ),
    ) );
    $header_args = array(
        'default-image'         => '',
        'default-text-color'    => '000',
        'width'                 => 220,
        'height'                => 125,
        'flex-height'           => true,
        'flex-width'            => true,
    );
    add_theme_support( 'custom-header', $header_args );

}
endif; // ninja_setup
add_action( 'after_setup_theme', 'ninja_setup' );

/**
 * Register widgetized area and update sidebar with default widgets.
 */
function ninja_widgets_init() {

    register_sidebar( array(
        'id'            => 'sidebar-widget',
        'name'          => esc_html__( 'Sidebar', 'ninja' ),
        'description'   => esc_html__( 'The default sidebar widget', 'ninja' ),
        'before_widget' => '<aside id="%1$s" class="widget %2$s">',
        'after_widget'  => '</aside>',
        'before_title'  => '<h3 class="widgettitle">',
        'after_title'   => '</h3>',
    ) );

    register_sidebar( array(
        'id'            => 'footer-form-widget',
        'name'          => esc_html__( 'Footer Form', 'ninja' ),
        'description'   => esc_html__( 'The footer form area', 'ninja' ),
        'before_widget' => '<div id="%1$s" class="widget %2$s">',
        'after_widget'  => '</div>',
        'before_title'  => '<h3 class="widgettitle">',
        'after_title'   => '</h3>',
    ) );

    register_sidebar( array(
        'id'            => 'footer-logo-widget',
        'name'          => esc_html__( 'Footer Logo', 'ninja' ),
        'description'   => esc_html__( 'The footer logo area', 'ninja' ),
        'before_widget' => '<div id="%1$s" class="widget %2$s">',
        'after_widget'  => '</div>',
        'before_title'  => '<h3 class="widgettitle">',
        'after_title'   => '</h3>',
    ) );

    register_sidebar( array(
        'id'            => 'footer-widget-1',
        'name'          => esc_html__( 'Footer Area 1', 'ninja' ),
        'description'   => esc_html__( 'The footer widget area 1', 'ninja' ),
        'before_widget' => '<div id="%1$s" class="widget %2$s">',
        'after_widget'  => '</div>',
        'before_title'  => '<h3 class="widgettitle">',
        'after_title'   => '</h3>',
    ) );

    register_sidebar( array(
        'id'            => 'footer-widget-2',
        'name'          => esc_html__( 'Footer Area 2', 'ninja' ),
        'description'   => esc_html__( 'The footer widget area 2', 'ninja' ),
        'before_widget' => '<div id="%1$s" class="widget %2$s">',
        'after_widget'  => '</div>',
        'before_title'  => '<h3 class="widgettitle">',
        'after_title'   => '</h3>',
    ) );

    register_sidebar( array(
        'id'            => 'floating-widget',
        'name'          => esc_html__( 'Floating Widget', 'ninja' ),
        'description'   => esc_html__( 'Floating Widget', 'ninja' ),
        'before_widget' => '<div id="%1$s" class="floating widget %2$s">',
        'after_widget'  => '</div>',
        'before_title'  => '<h3 class="widgettitle">',
        'after_title'   => '</h3>',
    ) );

    register_widget( 'ninja_social_widget' );
}
add_action( 'widgets_init', 'ninja_widgets_init' );

/**
 * Theme Widgets
 */
require_once(get_template_directory() . '/inc/widget-social.php');

/**
 * Unregister all unnecessary widgets 
 */
function ninja_unregister_default_widgets() {
    unregister_widget('WP_Widget_Pages');
    unregister_widget('WP_Widget_Calendar');
    unregister_widget('WP_Widget_Links');
    unregister_widget('WP_Widget_Meta');
    unregister_widget('WP_Widget_Recent_Comments');
    unregister_widget('WP_Widget_Recent_Posts');
    unregister_widget('WP_Widget_RSS');
    unregister_widget('WP_Widget_Tag_Cloud');

    // WOO WIDGETS
    unregister_widget( 'WC_Widget_Layered_Nav' );
    unregister_widget( 'WC_Widget_Layered_Nav_Filters' );
    unregister_widget( 'WC_Widget_Price_Filter' );
    unregister_widget( 'WC_Widget_Product_Tag_Cloud' );
    unregister_widget( 'WC_Widget_Recent_Reviews' );
    unregister_widget( 'WC_Widget_Recently_Viewed' );
    unregister_widget( 'WC_Widget_Top_Rated_Products' );
} 
add_action('widgets_init', 'ninja_unregister_default_widgets', 11);

/**
 * This function removes inline styles set by WordPress gallery.
 */
function ninja_remove_gallery_css( $css ) {
  return preg_replace( "#<style type='text/css'>(.*?)</style>#s", '', $css );
}
add_filter( 'gallery_style', 'ninja_remove_gallery_css' );

/**
 * Enqueue scripts and styles.
 */
function ninja_scripts() {

  // Bootstrap default CSS
  wp_enqueue_style( 'ninja-bootstrap', get_template_directory_uri() . '/inc/css/bootstrap.min.css' );

  // Font Awesome
  wp_enqueue_style( 'ninja-icons', '//use.fontawesome.com/releases/v5.4.2/css/all.css' );

  // Google Fonts
  wp_enqueue_style( 'ninja-fonts', '//fonts.googleapis.com/css?family=Montserrat:300,400,500,700,900');

  // Main theme stylesheet
  wp_enqueue_style( 'ninja-style', get_stylesheet_uri() );

  // Load jquery
  wp_enqueue_script('jquery');

  // Modernizr for better HTML5 and CSS3 support
  wp_enqueue_script('ninja-modernizr', get_template_directory_uri().'/inc/js/modernizr.min.js', array('jquery') );

  // Bootstrap default JS
  wp_enqueue_script('ninja-bootstrapjs', get_template_directory_uri().'/inc/js/bootstrap.min.js', array('jquery') );

  // Main theme related functions
  wp_enqueue_script( 'ninja-functions', get_template_directory_uri() . '/inc/js/functions.min.js', array('jquery') );

  // Smooth scroll script
  wp_enqueue_script( 'smooth-scroll-script', get_template_directory_uri() . '/inc/js/smooth.scroll.js', array('jquery') );

  // Custom style
  wp_enqueue_style( 'ninja-custom-style', get_template_directory_uri().'/css/custom.css' );

  // Custom scripts
  if ( is_page_template('page-thankyou.php') ) {
    wp_enqueue_script( 'thank-you-script', get_template_directory_uri() . '/js/tqcountdown.js', array('jquery') ,'1.', true);
  }
  //wp_enqueue_script( 'parallax-script', get_template_directory_uri() . '/js/parallax.js', array('jquery') );
  wp_enqueue_script( 'particle', '//cdn.jsdelivr.net/npm/particles.js@2.0.0/particles.min.js', array(), false, true );
  wp_enqueue_script( 'custom-script', get_template_directory_uri() . '/js/custom.js', array('jquery'), false, true );

}
add_action( 'wp_enqueue_scripts', 'ninja_scripts' );

function woocommerce_styling(){
    wp_enqueue_style( 'woocommerce-style', get_template_directory_uri().'/woocommerce/woocommerce.css' );
}

if ( in_array( 'woocommerce/woocommerce.php', apply_filters( 'active_plugins', get_option( 'active_plugins' ) ) ) ) {
    //add_action( 'wp_enqueue_scripts', 'woocommerce_styling' );
    //require get_template_directory() . '/woocommerce/woo-functions.php';
}

/**
 * Extra functions for this theme.
 */
require get_template_directory() . '/inc/navwalker.php'; // CUSTOM NAV WALKER
require get_template_directory() . '/inc/navigation.php'; // NAVIGATION RELATED FUNCTIONS
require get_template_directory() . '/inc/extras.php'; // THEME RELATED FUNCTIONS

/**
 * Default Excerpt Length
 */
function ninja_excerpt_length( $length ) {
    return 25;
}
add_filter( 'excerpt_length', 'ninja_excerpt_length' );

/**
 * Shortcodes in excerpt and widgets
 */
add_filter('the_excerpt', 'do_shortcode');
add_filter('widget_text','do_shortcode');

/**
 * Allow shortcodes inside Contact Form 7 forms
 */
function shortcode_wpcf7_form( $form ) {
    $form = do_shortcode( $form );
    return $form;
}
add_filter( 'wpcf7_form_elements', 'shortcode_wpcf7_form' );

/**
 * Create layout for ninja breadcrumbs
 */
function ninja_breadcrumbs(){

    if ( function_exists('bcn_display') ){
        $output = '';
        $output .= '<!--start breadcrumbs--><div class="breadcrumbs">';

        $container = set_ninja_container();
        $ninja_container = $container['container_type'];
        $breadcrumbs = bcn_display(true, true, false);

        if ( $ninja_container == 'container-fluid' ){
            $output .= '<div class="container"><div class="row"><div class="col-sm-12">';
            $output .= $breadcrumbs;
            $output .= '</div></div></div>';
        }else{
            $output .= $breadcrumbs;
        }
        
        $output .= '</div><!--end breadcrumbs-->';
        echo $output;
    }

}
add_action( 'ninja_breadcrumbs', 'ninja_breadcrumbs');

/**
 * Set pagenavi plugin as default navigation
 */
function ninja_navigation(){
    if ( function_exists('wp_pagenavi') ) { wp_pagenavi(); }
}
add_action( 'ninja_navigation', 'ninja_navigation');

/**
 * Remove archive type from title
 */
add_filter( 'get_the_archive_title', 'remove_prefix_archive_title' );
function remove_prefix_archive_title ($title) {
    if ( is_category() ) {
        $title = single_cat_title( '', false );
    } elseif ( is_tag() ) {
        $title = single_tag_title( '', false );
    } elseif ( is_author() ) {
        $title = '<span class="vcard">' . get_the_author() . '</span>' ;
    } elseif ( is_tax() ){
        $title = single_term_title( '', false );
    } elseif ( is_post_type_archive() ){
        $title = post_type_archive_title( '', false );
    }
    return $title;
}

/* Hide WP version strings from scripts and styles
* @return {string} $src
* @filter script_loader_src
* @filter style_loader_src
*/
function ninja_remove_wp_version_strings( $src ) {
    global $wp_version;
    parse_str(parse_url($src, PHP_URL_QUERY), $query);
    if ( !empty($query['ver']) && $query['ver'] === $wp_version ) {
        $src = remove_query_arg('ver', $src);
    }
    return $src;
}
add_filter( 'script_loader_src', 'ninja_remove_wp_version_strings' );
add_filter( 'style_loader_src', 'ninja_remove_wp_version_strings' );

/* Hide WP version strings from generator meta tag */
function ninja_remove_version() {
    return '';
}
add_filter('the_generator', 'ninja_remove_version');

/**
 * Grab all Toolset Types Custom Fields by Group ID or Group Slug
 */
function types_render_group($target_group) {
    global $post;
    require_once WPCF_EMBEDDED_INC_ABSPATH . '/fields.php';
     
    //by known group_id
    if(is_numeric($target_group)){
        return wpcf_admin_fields_get_fields_by_group($target_group);
    }
    //by known slug
    else{
        $groups = wpcf_admin_fields_get_groups();
        foreach($groups AS $group){
            if($group['slug'] == $target_group){
                return wpcf_admin_fields_get_fields_by_group($group['id']);
            }
        }
    }
    //check yourself, dude
    return false;
}

/**
 * Display page banner
 */
function display_page_banner() {

    if ( is_front_page() || is_page_template('page-thankyou.php') ) { return; }

    $output             = null;
    $item_atts          = [];
    $default_banner_url = wp_get_attachment_image_src( 41, 'full', false )[0];
    $title              = null;

    $banner_atts[] = 'background-image: url('.$default_banner_url.')';

    $output .= '<div class="page-banner bg-white">';
    $output .= '<div class="page-banner__inner"';
    if ( !empty($banner_atts) ) {
        $output .= 'style="'.implode("; ", $banner_atts).'"';
    }
    $output .= '>'; // close opening div

    $output .= '</div>';
    $output .= '</div>';

    echo $output;
}